import numpy as np
import matplotlib.pyplot as plt
from matplotlib.widgets import Button
import matplotlib.animation as animation
import seaborn as sns

# Configuration du style
sns.set_style("whitegrid")
plt.rcParams['figure.facecolor'] = 'white'
plt.rcParams['axes.facecolor'] = 'white'

# Paramètres initiaux
coupon_initial = 0.06  # Coupon 6%
yield_initial = 0.076  # Taux 7.6%
maturity_initial = 7   # Maturité 7 ans
face_value = 10000     # Nominal 10000€
step_initial = 10      # Pas de 10 pb

# 1. D'abord, définir la couleur initiale au début du code
COULEUR_POSITION_INITIALE =  'darkblue'

# Configuration du graphique - Longueur augmentée, hauteur inchangée
fig, ax = plt.subplots(figsize=(24, 10))  # Longueur 24, hauteur 10
plt.subplots_adjust(bottom=0.15, right=0.65, left=0.08, top=0.92)  # Réduction de right à 0.65

# Fonction de calcul du prix
def bond_price(coupon, y, maturity, face_value):
    C = coupon * face_value
    P = sum([C / (1+y)**t for t in range(1, maturity)]) + (C+face_value) / (1+y)**maturity
    return P

# Modification du calcul de la courbe Prix/Rendement
yields = np.linspace(0, 0.20, 100)  # Rendements de 0% à 20%
prices = [bond_price(coupon_initial, y, maturity_initial, face_value) for y in yields]

# Tracé initial
line_prix, = ax.plot(yields*100, prices, 'b-', lw=2, label='Courbe Prix/Rendement')
# 2. Modifier le tracé initial pour utiliser cette couleur
point_initial = ax.plot(
    yield_initial*100,
    bond_price(coupon_initial, yield_initial, maturity_initial, face_value),
    'o', ms=12, markerfacecolor='white',
    markeredgecolor=COULEUR_POSITION_INITIALE,  # Utilise la couleur définie
    markeredgewidth=2.5,
    label='Position initiale'
)

# Calcul initial des métriques pour le point de départ
price_initial = bond_price(coupon_initial, yield_initial, maturity_initial, face_value)
D_mac_initial = sum([t * coupon_initial * face_value / (1+yield_initial)**t for t in range(1, maturity_initial)])
D_mac_initial += maturity_initial * (coupon_initial * face_value + face_value) / (1+yield_initial)**maturity_initial
D_mac_initial /= price_initial
D_mod_initial = D_mac_initial / (1 + yield_initial)

# Style et configuration des axes
ax.set_xlabel('Rendement (%)', fontsize=14, fontweight='bold')
ax.set_ylabel('Prix', fontsize=14, fontweight='bold')
ax.set_title('Simulation Duration et Convexité Obligataire', fontsize=16, fontweight='bold', pad=20)
ax.tick_params(axis='both', labelsize=12)
ax.grid(True, alpha=0.3, linestyle='--')
ax.legend(fontsize=10)

# Ajustement des échelles et limites des axes
ax.set_xlim([0,18])  # De 0% à 20%
ax.set_ylim([2000, 15000])  # Ajustement automatique pour les prix correspondants

# Style amélioré pour une meilleure lisibilité
plt.rcParams['figure.dpi'] = 100  # Augmentation de la résolution
plt.rcParams['savefig.dpi'] = 100
plt.rcParams['font.size'] = 12
plt.rcParams['axes.linewidth'] = 2
plt.rcParams['grid.linewidth'] = 0.5

# Amélioration du style des grilles
ax.grid(True, which='major', linestyle='-', alpha=0.3)
ax.grid(True, which='minor', linestyle=':', alpha=0.2)
ax.minorticks_on()  # Activer les graduations mineures

# Texte d'information - Position maintenue
info_text = ax.text(1.08, 0.98, '', transform=ax.transAxes,
                    bbox=dict(facecolor='white', alpha=0.9, edgecolor='gray', boxstyle='round,pad=1'),
                    verticalalignment='top', fontsize=12, family='sans-serif')
info_text.set_color(COULEUR_POSITION_INITIALE)

# Variables pour l'animation
animation_active = False
current_step = 0
STEP_SIZE = 0.001  # 10 points de base
MAX_STEPS = 50  # Moins d'étapes mais plus visibles
ANIMATION_INTERVAL = 800  # Plus lent pour mieux suivre
direction = 1     # 1 pour hausse, -1 pour baisse
simulation_phase = 'hausse'  # 'hausse' ou 'baisse'
annot = [] # liste des annotations lors de l'animation


# Boutons - Position ajustée plus bas
ax_start = plt.axes([0.70, 0.30, 0.12, 0.05])  # y réduit à 0.25
ax_stop = plt.axes([0.70, 0.23, 0.12, 0.05])   # y réduit à 0.18
ax_reset = plt.axes([0.70, 0.16, 0.12, 0.05])  # y réduit à 0.11

# Style des boutons maintenu
btn_start = Button(ax_start, 'DÉMARRER >', color='#90EE90')
btn_stop = Button(ax_stop, 'PAUSE ||', color='#FFB6C6')
btn_reset = Button(ax_reset, 'RESET <', color='#ADD8E6')

# Style des boutons inchangé
for btn in [btn_start, btn_stop, btn_reset]:
    btn.label.set_fontsize(14)
    btn.label.set_weight('bold')

# 3. Dans la fonction update_animation, modifier la ligne de réaffichage du point initial
def update_animation(frame):
    global current_step, animation_active
    if not animation_active:
        return line_prix,
    
    # Nettoyage des éléments précédents sauf le point initial
    for artist in ax.lines[3:]:  # On commence à 3 pour garder la courbe et le point initial
        artist.remove()
    for collection in ax.collections:
        collection.remove()
    
    # Réaffichage du point initial avec la même apparence
    ax.plot(yield_initial*100, price_initial, 'o', 
            ms=12, 
            markerfacecolor='white',
            markeredgecolor=COULEUR_POSITION_INITIALE,
            markeredgewidth=2.5,
            zorder=5)
    
    # Calcul des rendements
    yield_up = yield_initial + (current_step * STEP_SIZE)
    yield_down = yield_initial - (current_step * STEP_SIZE)
    
    # Calcul des prix réels
    price_up = bond_price(coupon_initial, yield_up, maturity_initial, face_value)
    price_down = bond_price(coupon_initial, yield_down, maturity_initial, face_value)
    
    # Calcul des prix approximés par la duration (approximation linéaire)
    price_duration_up = price_initial * (1 - D_mod_initial * (yield_up - yield_initial))
    price_duration_down = price_initial * (1 - D_mod_initial * (yield_down - yield_initial))
    
    # Tracé des tangentes depuis le point initial
    tangent = price_initial - D_mod_initial * price_initial * (yields - yield_initial)
    
    # Points pour montrer la différence entre prix réel et approximation duration
    ax.plot([yield_up*100], [price_up], 'ro', ms=8, label='Prix réel (hausse)')
    ax.plot([yield_up*100], [price_duration_up], 'rx', ms=8, label='Approx. duration (hausse)')
    ax.plot([yield_down*100], [price_down], 'go', ms=8, label='Prix réel (baisse)')
    ax.plot([yield_down*100], [price_duration_down], 'gx', ms=8, label='Approx. duration (baisse)')
    
    # Lignes verticales pour montrer l'écart
    ax.vlines(yield_up*100, price_duration_up, price_up, colors='red', linestyles=':', alpha=0.5)
    ax.vlines(yield_down*100, price_duration_down, price_down, colors='green', linestyles=':', alpha=0.5)
    
    # Zones de convexité avec motifs différents
    ax.fill_between(yields*100, prices, tangent,
                    where=(yields > yield_initial),
                    color='red', alpha=0.15, hatch='xxx', 
                    label='Zone de Convexité (Hausse)')
    ax.fill_between(yields*100, prices, tangent,
                    where=(yields < yield_initial),
                    color='green', alpha=0.15, hatch='///', 
                    label='Zone de Convexité (Baisse)')
    
    # Mise à jour du texte avec l'erreur d'approximation
    variation_bp = current_step * 10
    erreur_hausse = ((price_up - price_duration_up) / price_initial) * 100
    erreur_baisse = ((price_down - price_duration_down) / price_initial) * 100


    info_text.set_text('')

    # Texte principal dans info_text
    info_text.set_text(
        "POSITION INITIALE\n"
        "------------------------\n"
        f"Taux: {yield_initial*100:.2f}%\n"
        f"Prix: {price_initial:.0f} XOF\n"
        f"Duration Modifiée: {D_mod_initial:.2f}"
    )

    # Liste des annotations pour la hausse des taux avec un espacement réduit
    annotations_hausse = [
        (f"HAUSSE DES TAUX (+{variation_bp} pb)", 0.78, 'red'),
        (f"Taux de marché: {yield_up*100:.2f}%", 0.74, 'red'),  # Réduit l'espacement
        (f"Prix réel: {price_up:.0f} XOF", 0.70, 'red'),        # Réduit l'espacement
        (f"Prix prévu (Duration): {price_duration_up:.0f} XOF", 0.66, 'red'),  # Réduit l'espacement
        (f"Erreur Duration: {erreur_hausse:.2f}%", 0.62, 'red'),  # Réduit l'espacement
        ("→ La Duration surestime la baisse", 0.58, 'red'),
        ("- - - - - - - - - - - - - - - - - - - - - -", 0.54, 'black')             # Réduit l'espacement
    ]

    # Liste des annotations pour la baisse des taux avec un espacement réduit
    annotations_baisse = [
        (f"BAISSE DES TAUX (-{variation_bp} pb)", 0.50, 'green'),  # Réduit l'espacement
        (f"Taux de marché: {yield_down*100:.2f}%", 0.46, 'green'),  # Réduit l'espacement
        (f"Prix réel: {price_down:.0f} XOF", 0.42, 'green'),        # Réduit l'espacement
        (f"Prix prévu (Duration): {price_duration_down:.0f} XOF", 0.38, 'green'),  # Réduit l'espacement
        (f"Erreur Duration: {erreur_baisse:.2f}%", 0.34, 'green'),  # Réduit l'espacement
        ("→ La Duration sous-estime la hausse", 0.30, 'green')      # Réduit l'espacement
    ]

    
    # Fonction pour ajouter les annotations avec contours
    def add_annotations(annotations, annot):
        for text, y_pos, color in annotations:
            annot.append(ax.annotate(
                text,
                xy=(1.05, y_pos),  # Position x ajustée pour l'alignement à gauche
                xycoords='axes fraction',
                fontsize=12,
                family='sans-serif',
                color=color,
                bbox=dict(facecolor='white', alpha=0.9, edgecolor='none', boxstyle='round,pad=0.3')  # Contour autour du texte
            ))
    
    # Ajouter les annotations pour la hausse et la baisse
    add_annotations(annotations_hausse,annot)
    add_annotations(annotations_baisse,annot)

    ax.legend(loc='upper right', bbox_to_anchor=(0.98, 0.98))
    
    current_step += 1
    if current_step > MAX_STEPS:
        current_step = 0
        animation_active = False
    
    return line_prix,

def start_animation(event):
    global animation_active
    animation_active = True

def stop_animation(event):
    global animation_active
    animation_active = False

def reset_animation(event):
    global current_step, simulation_phase, animation_active, annot 
    current_step = 0
    simulation_phase = 'hausse'
    animation_active = False
    # Nettoyage des éléments graphiques
    for artist in ax.lines[2:]:
        artist.remove()
    for collection in ax.collections:
        collection.remove()
    # Réinitialisation du texte
    info_text.set_text('')

    for annotation in annot:
        annotation.remove()  # Supprimer chaque annotation
    annot.clear()



    plt.draw()

btn_start.on_clicked(start_animation)
btn_stop.on_clicked(stop_animation)
btn_reset.on_clicked(reset_animation)

# Style de la courbe principale
line_prix.set_linewidth(4)
line_prix.set_color('#000080')  # Bleu marine plus profond
line_prix.set_zorder(2)

# Style du point initial - Définition d'une couleur consistante
COULEUR_POSITION_INITIALE = 'darkblue'  # ou '#00008B'

# Point initial plus visible avec la couleur harmonisée
point_initial[0].set_markersize(12)
point_initial[0].set_markerfacecolor('white')
point_initial[0].set_markeredgecolor('black')  # Remise en noir
point_initial[0].set_markeredgewidth(2.5)
point_initial[0].set_zorder(5)  # Toujours au premier plan

# Animation
anim = animation.FuncAnimation(fig, update_animation, 
                             interval=ANIMATION_INTERVAL,  # 500ms entre chaque frame
                             blit=False,
                             cache_frame_data=False)

plt.show()